<?php

namespace App\Services\FollowUp;

use App\Models\FollowUp;
use App\Models\User;
use App\Models\Lead;
use App\Services\Lead\LeadService;

class FollowUpService
{
    public function __construct(
        protected LeadService $leadService
    ) {}

    /**
     * Create follow-up
     */
    public function create(User $user, array $data): FollowUp
    {
        $followUp = FollowUp::create([
            'user_id' => $user->id,
            ...$data,
        ]);

        $this->leadService->updateNextFollowUp($followUp->lead);

        return $followUp;
    }

    /**
     * Mark follow-up as done
     */
    public function markDone(FollowUp $followUp): FollowUp
    {
        $followUp->update([
            'status' => 'done',
            'completed_at' => now(),
        ]);

        $this->leadService->updateNextFollowUp($followUp->lead);

        return $followUp;
    }

    /**
     * Skip follow-up
     */
    public function skip(FollowUp $followUp): FollowUp
    {
        $followUp->update([
            'status' => 'skipped',
        ]);

        $this->leadService->updateNextFollowUp($followUp->lead);

        return $followUp;
    }

    /**
     * Reschedule follow-up
     */
    public function reschedule(FollowUp $followUp, string $scheduledAt): FollowUp
    {
        $followUp->update([
            'scheduled_at' => $scheduledAt,
            'status' => 'pending',
        ]);

        $this->leadService->updateNextFollowUp($followUp->lead);

        return $followUp;
    }

    public function snooze(FollowUp $followUp, string $type): FollowUp
    {
        $now = Carbon::now();

        $newSchedule = match ($type) {
            '15m' => $now->copy()->addMinutes(15),
            '1h' => $now->copy()->addHour(),
            'tomorrow' => $now->copy()->addDay()->setTime(10, 0),
            'next_week' => $now->copy()->addWeek()->setTime(10, 0),
            default => $now->copy()->addMinutes(15),
        };

        $followUp->update([
            'scheduled_at' => $newSchedule,
            'status' => 'pending',
        ]);

        $this->leadService->updateNextFollowUp($followUp->lead);

        return $followUp;
    }
}
