<?php

namespace App\Http\Controllers\API\V1;

use App\Http\Controllers\Controller;
use App\Models\FollowUp;
use Illuminate\Http\Request;
use App\Http\Requests\FollowUp\StoreFollowUpRequest;
use App\Http\Requests\FollowUp\RescheduleFollowUpRequest;
use App\Http\Requests\FollowUp\SnoozeFollowUpRequest;
use App\Services\FollowUp\FollowUpService;
use Carbon\Carbon;

class FollowUpController extends Controller
{
    public function __construct(
        protected FollowUpService $followUpService
    ) {}

    /**
     * Today's follow-ups
     */
    public function today(Request $request)
    {
        $items = FollowUp::with('lead')->where('user_id', $request->user()->id)
            ->today()
            ->get();

        return response()->json([
            'success' => true,
            'data' => $items,
            'error' => null,
        ]);
    }

    /**
     * Upcoming follow-ups
     */
    public function upcoming(Request $request)
    {
        $user = $request->user()->load('subscription');

        $query = FollowUp::with('lead')
            ->where('user_id', $user->id)
            ->where('scheduled_at', '>', now())
            ->orderBy('scheduled_at');

        if ($user->plan !== 'pro') {
            $query->limit(1);
        }

        return response()->json([
            'success' => true,
            'data' => $query->get(),
            'error' => null,
        ]);
    }

    /**
     * Create follow-up
     */
    public function store(StoreFollowUpRequest $request)
    {
        $followUp = $this->followUpService->create(
            $request->user(),
            $request->validated()
        );

        return response()->json([
            'success' => true,
            'data' => $followUp,
            'error' => null,
        ], 201);
    }

    /**
     * Mark follow-up as done
     */
    public function markDone(FollowUp $followUp)
    {
        $this->authorize('update', $followUp);

        $followUp = $this->followUpService->markDone($followUp);

        // ✅ Load lead to return name/phone/email for UI modal
        $followUp->load('lead');

        return response()->json([
            'success' => true,
            'data' => $followUp,
            'error' => null,
        ]);
    }

    /**
     * Skip follow-up
     */
    public function skip(FollowUp $followUp)
    {
        $this->authorize('update', $followUp);

        $followUp = $this->followUpService->skip($followUp);

        return response()->json([
            'success' => true,
            'data' => $followUp,
            'error' => null,
        ]);
    }

    /**
     * Reschedule follow-up
     */
    // public function reschedule(
    //     RescheduleFollowUpRequest $request,
    //     FollowUp $followUp
    // ) {
    //     $this->authorize('update', $followUp);

    //     $followUp = $this->followUpService->reschedule(
    //         $followUp,
    //         $request->scheduled_at
    //     );

    //     return response()->json([
    //         'success' => true,
    //         'data' => $followUp,
    //         'error' => null,
    //     ]);
    // }

    public function reschedule(
        RescheduleFollowUpRequest $request,
        FollowUp $followUp
    ) {
        $this->authorize('update', $followUp);

        $followUp = $this->followUpService->reschedule(
            $followUp,
            $request->validated()['scheduled_at']
        );

        return response()->json([
            'success' => true,
            'data' => $followUp,
            'error' => null,
        ]);
    }

    public function snooze(SnoozeFollowUpRequest $request, FollowUp $followUp)
    {
        $this->authorize('update', $followUp);

        $followUp = $this->followUpService->snooze(
            $followUp,
            $request->validated()['type']
        );

        return response()->json([
            'success' => true,
            'data' => $followUp,
            'error' => null,
        ]);
    }

    public function overdue(Request $request)
    {
        $user = $request->user();

        $query = FollowUp::with('lead')
            ->where('user_id', $user->id)
            ->overdue()
            ->orderBy('scheduled_at', 'asc');

        // ✅ Free user lock (show only 1)
        if ($user->plan !== 'pro') {
            $query->limit(1);
        }

        return response()->json([
            'success' => true,
            'data' => $query->get(),
            'error' => null,
        ]);
    }

    public function overdueCount(Request $request)
    {
        $count = FollowUp::where('user_id', $request->user()->id)
            ->overdue()
            ->count();

        return response()->json([
            'success' => true,
            'data' => [
                'count' => $count,
            ],
            'error' => null,
        ]);
    }
}
